-- Setup the loader.
VMFLoader = {};

-- Load includes.
_OpenScript( "vmfloader/luavfs.lua" );
_OpenScript( "vmfloader/tokenreader.lua" );
_OpenScript( "vmfloader/parser.lua" );
_OpenScript( "vmfloader/filter.lua" );
_OpenScript( "vmfloader/utility.lua" );
_OpenScript( "vmfloader/admin.lua" );
_OpenScript( "vmfloader/folders.lua" );
_OpenScript( "vmfloader/spawnmenu.lua" );
_OpenScript( "vmfloader/addons.lua" );
_OpenScript( "vmfloader/commands.lua" );
_OpenScript( "vmfloader/config.lua" );
_OpenScript( "vmfloader/lua.lua" );

-- Requested by aVoN.
VMFLoader.version = 0.15;

-- Globals.
VMFLoader.FixupVectors = { "origin", "springaxis", "attachpoint", "hingeaxis", "point0", "point1", "point2", "position2", "slideaxis" };
VMFLoader.FixupAngles = { "angles", "movedir" };
VMFLoader.ActivateEntities =    {
                                    { class = "phys_", partial = true },
                                    { class = "path_", partial = true },
                                };
VMFLoader.PreviewEntities = {
                                { class = "npc_", partial = true },
                                { class = "item_", partial = true },
                                { class = "weapon_", partial = true },
                                { class = "prop_", partial = true },
                                { class = "phys_magnet", partial = false },
                                { class = "combine_mine", partial = false },
                                { class = "physics_", partial = true },
                                { class = "move_rope", partial = false },
                                { class = "keyframe_rope", partial = false },
                            };
VMFLoader.SpawnCounter = 0;

-- Player information.
VMFLoader.Players = {};
for player = 1, _MaxPlayers() do
    VMFLoader.Players[player] = {};
    
    -- Administration.
    VMFLoader.Players[player].last_spawn = _CurTime();
    VMFLoader.Players[player].spawn_count = 0;
    
    -- VMF Undo.
    VMFLoader.Players[player].last_vmf_spawned = {};
    
    -- Preview.
    VMFLoader.Players[player].spawn_angle = 0;
    VMFLoader.Players[player].spawn_height = 0;
    VMFLoader.Players[player].in_preview = false;
    VMFLoader.Players[player].preview_entity = nil;
    VMFLoader.Players[player].entity_list = {};
    VMFLoader.Players[player].preview_file = "";
    VMFLoader.Players[player].preview_entity_position = vector3( 0, 0, 0 );
    VMFLoader.Players[player].preview_entity_angle = vector3( 0, 0, 0 );
end

-- Find keyvalue.
function VMFLoader:FindKV( keys, keyname )
    for _, kv in keys do
        if( string.lower( keyname ) == string.lower( kv.key ) ) then
            return kv;
        end
    end
    return nil;
end

-- Find entity.
function VMFLoader:FindEntity( entities, classname )
    for key, entity in entities do
        local clsname = VMFLoader:FindKV( entity.keyvalues, "classname" );
        if( clsname and string.lower( clsname.value ) == string.lower( classname ) ) then
            return key, entity;
        end
    end
    
    return nil, nil;
end

-- Fixup vector.
function VMFLoader:FixupVector( targetposition, targetrotation, position, targetheight )
    -- Move to target position.
    local vNewPosition = vecAdd( position, targetposition );
    vNewPosition.z = 0;
    
    -- Figure out the angle between the 2 points.
    local angle = VMFLoader:AngleBetweenPoints( targetposition, vNewPosition ) + targetrotation;
    
    -- Clamp.
    if( angle > 360 ) then
        angle = angle - 360;
    end
    
    -- Get the distance from the target position.
    local dist = math.sqrt( position.x * position.x + position.y * position.y );
    
    -- Rotate around targetposition.
    vNewPosition.x = targetposition.x + ( dist * math.cos( math.rad( angle ) ) );
    vNewPosition.y = targetposition.y + ( dist * math.sin( math.rad( angle ) ) );
    vNewPosition.z = targetposition.z + position.z + targetheight;
    
    -- return the fixed vector.
    return vNewPosition;
end

-- Fixup angle.
function VMFLoader:FixupAngle( currentangle, targetrotation )
    local vAngles = vector3( currentangle.x, currentangle.y + targetrotation, currentangle.z );
    if( vAngles.y > 360 ) then
        vAngles.y = vAngles.y - 360;
    end
    return vAngles;
end

-- Load entity list.
function VMFLoader:LoadEntityList( player, entitylist, absolute, lua_keys )
    -- Trace to see where to spawn.
    _TraceSetCollisionGroup( COLLISION_GROUP_DEBRIS );
    PlayerLookTrace( player, 4096 );
    
    -- Table to store entity id's for spawning later.
    local entityids = {};
    
    -- Loop through entities.
    for _, entity in entitylist do
        -- Run Keyvalue event.
        Addon:CallEvent( player, Addon.EVENT_KEYVALUES, entity.keyvalues );
        
        -- Run connection event.
        Addon:CallEvent( player, Addon.EVENT_CONNECTIONS, entity.connections );
        
        -- If we have a classname create.
        local classname = VMFLoader:FindKV( entity.keyvalues, "classname" );
        if( classname ) then
            -- Create
            local iEntity = _EntCreate( classname.value );
            
            -- Add keyvalues.
            for _, kv in entity.keyvalues do
                if( absolute ) then
                    _EntSetKeyValue( iEntity, kv.key, kv.value );
                else
                    -- Fix positions.
                    for _, fixup in VMFLoader.FixupVectors do
                        if( fixup == kv.key ) then
                            kv.value = vecString( VMFLoader:FixupVector( _TraceEndPos(), VMFLoader.Players[player].spawn_angle, VMFLoader:StringToVector( kv.value ), VMFLoader.Players[player].spawn_height ) );
                        end
                    end
                    
                    -- Fix angles.
                    for _, fixup in VMFLoader.FixupAngles do
                        if( fixup == kv.key ) then
                            kv.value = vecString( VMFLoader:FixupAngle( VMFLoader:StringToVector( kv.value ), VMFLoader.Players[player].spawn_angle ) );
                        end
                    end
                    
                    -- Set keyvalue like normal.
                    _EntSetKeyValue( iEntity, kv.key, kv.value );
                end
            end
            
            -- Add outputs.
            for _, kv in entity.connections do
                _EntFire( iEntity, "AddOutput", kv.key .. " " .. kv.value, 0 );
            end
            
            -- Run entity created event.
            Addon:CallEvent( player, Addon.EVENT_ENTITY_CREATED, iEntity );
            
            -- Store entity id.
            local tEntity = {};
            tEntity.entity = iEntity;
            tEntity.keyvalues = entity.keyvalues;
            tEntity.connections = entity.connections;
            tEntity.parentname = VMFLoader:FindKV( entity.keyvalues, "parentname" );
            table.insert( entityids, tEntity );
        end
    end
    
    -- Spawn entities.
    for _, entity in entityids do
        -- Spawn.
        _EntSpawn( entity.entity );
        
        -- Run spawn event.
        Addon:CallEvent( player, Addon.EVENT_SPAWN_ENTITY, entity.entity, entity.keyvalues, entity.connections );
    end
    
    -- Activate entities.
    for _, entity in entityids do
        _EntActivate( entity.entity );
    end
    
    -- Parent entities.
    for _, entity in entityids do
        -- Quick hack to see if it works or not.
        -- Comment out later.
        --[[if( _EntGetType( entity.entity ) ~= "func_useableladder" ) then
            local angles = VMFLoader:FindKV( entity.keyvalues, "angles" );
            if( angles ) then
                _EntSetAngAngle( entity.entity, VMFLoader:StringToVector( angles.value ) );
            end
            local origin = VMFLoader:FindKV( entity.keyvalues, "origin" );
            if( origin ) then
                _EntSetPos( entity.entity, VMFLoader:StringToVector( origin.value ) );
            end
        end]]
        
        -- Parent.
        if( entity.parentname ~= nil ) then
            _EntFire( entity.entity, "SetParent", entity.parentname.value, 0 );
        end
    end
    
    -- Store entity list in the vmf_undo buffer.
    local temp = {};
    for _, entity in entityids do
        table.insert( temp, entity.entity );
    end
    table.insert( VMFLoader.Players[player].last_vmf_spawned, temp );
    
    -- Call lua.
    if( VMFLoader.EmbeddedLua ) then
        VMFLua.player = player;
        VMFLua.spawncount = VMFLoader.SpawnCounter;
        for _, luacode in lua_keys do
            local code = string.gsub( luacode.value, "___", "_" .. tostring( VMFLua.spawncount ) .. "_"  );
            code = string.gsub( code, "\\r\\n", "\r\n" );
            _Msg( code );
            _RunString( code );
        end
    end
    
    -- Pass a list of all created entities to the addons.
    Addon:CallEvent( player, Addon.EVENT_SPAWNING_COMPLETE, entityids );
end

-- Valid preview entity?
function VMFLoader:IsValidPreviewEntity( classname )
    for _, valid in VMFLoader.PreviewEntities do
        if( valid.partial ) then
            if( string.find( string.lower( classname ), string.lower( valid.class ) ) ) then
                return true;
            end
        else
            if( string.lower( classname ) == string.lower( valid.class ) ) then
                return true;
            end
        end
    end

    return false;
end

-- Activate entity?
function VMFLoader:ShouldActivateEntity( entity )
    for _, valid in VMFLoader.ActivateEntities do
        if( valid.partial ) then
            if( string.find( string.lower( _EntGetType( entity ) ), string.lower( valid.class ) ) ) then
                return true;
            end
        else
            if( string.lower( _EntGetType( entity ) ) == string.lower( valid.class ) ) then
                return true;
            end
        end
    end

    return false;
end

-- Load entity list.
function VMFLoader:LoadPreview( player, filename, entitylist )   
    -- Table to store entity id's for spawning later.
    local entityids = {};
    
    -- Loop through entities.
    for _, entity in entitylist do
        -- If we have a classname create.
        local classname = VMFLoader:FindKV( entity.keyvalues, "classname" );
        if( classname and VMFLoader:IsValidPreviewEntity( classname.value ) ) then
            -- Create
            local iEntity = _EntCreate( classname.value );
            
            -- Add keyvalues.
            for _, kv in entity.keyvalues do
                _EntSetKeyValue( iEntity, kv.key, kv.value );
            end
            
            -- Add outputs.
            for _, kv in entity.connections do
                _EntFire( iEntity, "AddOutput", kv.key .. " " .. kv.value, 0 );
            end
            
            -- Store entity id.
            local tEntity = {};
            tEntity.entity = iEntity;
            tEntity.parentname = VMFLoader:FindKV( entity.keyvalues, "parentname" );
            table.insert( entityids, tEntity );
        end
    end
    
    -- Spawn entities.
    for _, entity in entityids do
        _EntSpawn( entity.entity );
    end
    
    -- Activate entities.
    for _, entity in entityids do
        if( VMFLoader:ShouldActivateEntity( entity.entity ) ) then
            _EntActivate( entity.entity );
        end
    end
    
    -- Parent everything to the first entity, make nonsolid, make transparent and save as preview_entity.
    VMFLoader:CleanupPreview( player );
    if( table.getn( entityids ) > 0 ) then
        local firstentity = entityids[1].entity;
        for _, entity in entityids do
            -- Parent.
            _EntSetParent( entity.entity, firstentity );
            
            -- Make nonsolid.
            _EntSetSolid( entity.entity, SOLID_NONE );
            _EntSetMoveType( entity.entity, MOVETYPE_NONE );
            _EntSetCollisionGroup( entity.entity, COLLISION_GROUP_NONE );

            -- Make transparent.
            if( string.find( _EntGetType( entity.entity ), "func_" ) == nil ) then
                _EntFire( entity.entity, "addoutput", "rendermode 4", 0 );
            else
                _EntFire( entity.entity, "addoutput", "rendermode 10", 0 );
            end
            _EntFire( entity.entity, "addoutput", "renderamt 80", 0 );
            
            -- Push into entity list
            table.insert( VMFLoader.Players[player].entity_list, entity.entity );
        end
        
        --Save.
        VMFLoader.Players[player].in_preview = true;
        VMFLoader.Players[player].preview_entity = firstentity;
        VMFLoader.Players[player].preview_entity_position = _EntGetPos( firstentity );
        VMFLoader.Players[player].preview_entity_angle = _EntGetAngAngle( firstentity );
    end
    
    -- Save filename.
    VMFLoader.Players[player].preview_file = filename;
    
    -- Spawn tool?
    if( VMFLoader.UseSpawnTool ) then
        -- Give the player the tool
        _PlayerGiveSWEP( player, "vmfloader/spawntool.lua" );
        
        -- Make them draw it.
        _PlayerSelectWeapon( player, "weapon_spawntool" );
    end
    
    -- Show context menu.
    _player.SetContextMenu( player, "vmfloader" );
end

-- Preview think.
function VMFLoader.PreviewThink( )
    -- Go through all players, and update their preview entities.
    for player = 1, _MaxPlayers() do
        if( _PlayerInfo( player, "connected" ) and _PlayerInfo( player, "alive" ) and VMFLoader.Players[player].in_preview ) then
            if( _EntExists( VMFLoader.Players[player].preview_entity ) ) then
                -- Move.
                _TraceSetCollisionGroup( COLLISION_GROUP_DEBRIS );
                PlayerLookTrace( player, 4096 );
                
                -- Move entity.
                local vDestination = VMFLoader:FixupVector( _TraceEndPos(), VMFLoader.Players[player].spawn_angle, VMFLoader.Players[player].preview_entity_position, VMFLoader.Players[player].spawn_height );
                _EntSetPos( VMFLoader.Players[player].preview_entity, vDestination );
                
                -- Rotate entity.
                local vAngle = VMFLoader:FixupAngle( VMFLoader.Players[player].preview_entity_angle, VMFLoader.Players[player].spawn_angle );
                _EntSetAngAngle( VMFLoader.Players[player].preview_entity, vAngle );
            end
        end
    end
end
AddThinkFunction( VMFLoader.PreviewThink );

-- Cleanup preview.
function VMFLoader:CleanupPreview( player )
    for _, entity in VMFLoader.Players[player].entity_list do
        if( _EntExists( entity ) ) then
            _EntRemove( entity );
        end
    end
    
    VMFLoader.Players[player].entity_list = {};
end

-- Run name filter.
function VMFLoader:IsNameStrongestMatch( name, search, namelist )
    local strongest_match = true;
    for _, _name in namelist do
        if( _name ~= name and string.find( search, _name ) ~= nil and string.len( _name ) > string.len( name ) ) then
            strongest_match = false;
        end
    end
    return strongest_match;
end

function VMFLoader:RunNameFilter( entities )
    -- Find entity names.
    local targetnames = {};
    for _, entity in entities do
        local targetname = VMFLoader:FindKV( entity.keyvalues, "targetname" );
        if( targetname ~= nil ) then
            local exists = false;
            for _, name in targetnames do
                if( string.lower( name ) == string.lower( targetname.value ) ) then
                    exists = true;
                end
            end
            
            if( not exists ) then
                if( string.find( targetname.value, "_unique" ) == nil ) then
                    table.insert( targetnames, targetname.value );
                end
            end
        end
    end
    
    -- Append spawn value.. After talking with aVoN he got me thinking :/
    -- So here is my edition of a faster version.
    for _, name in targetnames do
        for _, entity in entities do
            -- Replace keyvalues.
            for _, kv in entity.keyvalues do
                if( name == kv.value ) then 
                    kv.value = tostring( VMFLoader.SpawnCounter ) .. "__" .. name;
                end
            end
            
            -- Replace connections.
            for _, kv in entity.connections do
                if( VMFLoader:IsNameStrongestMatch( name, kv.value, targetnames ) ) then
                    kv.value = string.gsub( kv.value, name, tostring( VMFLoader.SpawnCounter ) .. "__" .. name );
                end
            end
        end
    end
end

-- Show contact.
function VMFLoader:ShowInformation( player, global_entity, filename )
    --- Bail out.
    if( not global_entity or not VMFLoader.ShowMetaData ) then
        return;
    end

    -- Show author information if it exists.
    local author = VMFLoader:FindKV( global_entity.keyvalues, "author" );
    local author_contact = VMFLoader:FindKV( global_entity.keyvalues, "author_contact" );
    local description = VMFLoader:FindKV( global_entity.keyvalues, "description" );
    
    -- Print.
    _PrintMessage( player, HUD_PRINTCONSOLE, "\n\nVMF Information\n========================================\n" );
    _PrintMessage( player, HUD_PRINTCONSOLE, "Filename: " .. filename .. "\n" );
    if( author ) then
        _PrintMessage( player, HUD_PRINTCONSOLE, "Author: " .. author.value .. "\n" );
    end
    if( author_contact ) then
        _PrintMessage( player, HUD_PRINTCONSOLE, "Author Contact: " .. author_contact.value .. "\n" );
    end
    if( description ) then
        _PrintMessage( player, HUD_PRINTCONSOLE, "Description: " .. string.gsub( description.value, "/n", "\n" ) .. "\n" );
    end
    _PrintMessage( player, HUD_PRINTCONSOLE, "========================================\n\n\n" );
end

-- Load VMF.
function VMFLoader.LoadVMF( player, filename )
    -- Arguments.
    if( filename == "" ) then
        return;
    end

    -- Check folder for admin.
    local allow_passage = true;
    for _, folder in Folder:GetFolders() do
        if( string.find( filename, folder.path ) ~= nil ) then
            if( folder.access ~= "" ) then
                if( not Admin:CheckAccess( player, folder.access ) ) then
                    allow_passage = false;
                end
            end 
        end
    end
    if( not allow_passage ) then
        return;
    end
    
    -- Should we even load this file check with addons.
    if( Addon:CallEvent( player, Addon.EVENT_LOAD_FILE, filename ) ) then
        return;
    end
    
    -- Increment the counter.
    VMFLoader.SpawnCounter = VMFLoader.SpawnCounter + 1;
    
    -- Read entities.
    local entities, world_keys, lua_keys = VMFParser:parse( filename );
    
    -- Search for a vmf_global entity.
    local global_key, global_entity = VMFLoader:FindEntity( entities, "vmf_global" );
    if( global_key and global_entity ) then
        -- Remove, we don't want to spawn this... It would be bad.
        table.remove( entities, global_key );
        
        -- Pass meta data to the addon system.
        Addon:CallEvent( player, Addon.EVENT_META_DATA, global_entity.keyvalues );
    end
    
    -- One time legacy name disable?
    local legacy_naming = VMFLoader:FindKV( world_keys, "legacy_naming" );
    if( global_entity ) then
        legacy_naming = VMFLoader:FindKV( global_entity.keyvalues, "legacy_naming" );
    end
    
    -- Run name filter.
    if( not VMFLoader.LegacyNaming and not ( legacy_naming and legacy_naming.value == "1" ) ) then
        VMFLoader:RunNameFilter( entities );
    end
    
    -- Alert addons the load was completed.
    if( Addon:CallEvent( player, Addon.EVENT_LOAD_COMPLETE, entities, world_keys, lua_keys ) ) then
        return;
    end
    
    -- Is this vmf absolute?
    local absolute = VMFLoader:FindKV( world_keys, "absolute" );
    if( global_entity ) then
        absolute = VMFLoader:FindKV( global_entity.keyvalues, "absolute" );
    end
    
    if( absolute and absolute.value == "1" ) then
        -- Load normally.
        VMFLoader:LoadEntityList( player, entities, true, lua_keys );
        VMFLoader:ShowInformation( player, global_entity, filename );
    else
        -- Preview disabled?
        local preview = VMFLoader:FindKV( world_keys, "nopreview" );
        if( global_entity ) then
            preview = VMFLoader:FindKV( global_entity.keyvalues, "nopreview" );
        end
        
        if( preview and preview.value == "1" ) then
            -- Load normally.
            VMFLoader:LoadEntityList( player, entities, false, lua_keys );
            VMFLoader:ShowInformation( player, global_entity, filename );
        else
            -- Load preview.
            if( VMFLoader.Players[player].in_preview and VMFLoader.Players[player].preview_file == filename ) then
                -- Clear vmf?
                if( VMFLoader.ClearVMF ) then
                    VMFLoader.Players[player].in_preview = false;
                    VMFLoader.Players[player].preview_entity = nil;
                    VMFLoader:CleanupPreview( player );
                end
                
                -- Load vmf.
                VMFLoader:LoadEntityList( player, entities, false, lua_keys );
                VMFLoader:ShowInformation( player, global_entity, filename );
            else
                VMFLoader:LoadPreview( player, filename, entities );
            end
        end
    end
end
CONCOMMAND( "loadvmf", VMFLoader.LoadVMF );
