
-- LuaVFS - Lua Virtual File System
-----------------------------------------
LuaVFS = {};
LuaVFS.Version = 0.14;
LuaVFS.Mounted = {};

-- File operations.

if( not oldExists ) then
    oldExists = _file.Exists;
end
function _file.Exists( filename )
    local exists = oldExists( filename );
    if( not exists ) then
        for _, system in LuaVFS.Mounted do
            if( system:exists( filename ) ) then
                exists = true;
            end
        end
    end
    return exists;
end

-- Is dir.
if( not oldIsDir ) then
    oldIsDir = _file.IsDir;
end
function _file.IsDir( dir )
    local isdir = oldIsDir( dir );
    if( not isdir ) then
        for _, system in LuaVFS.Mounted do
            if( system:isDirectory( dir ) ) then
                isdir = true;
            end
        end
    end
    return isdir;
end

if( not oldRead ) then
    oldRead = _file.Read;
end
function _file.Read( filename )
    -- Check if this file exists, if it does read normally, if it doesn't attempt to read from a mounted system.
    if( oldExists( filename ) ) then
        return oldRead( filename );
    else
        for _, system in LuaVFS.Mounted do
            if( system:exists( filename ) ) then
                return system:readFile( filename );
            end
        end
    end
    return "";
end

if( not oldDelete ) then
    oldDelete = _file.Delete;
end
function _file.Delete( filename )
    -- Check if this file exists, if it does delete normally, if it doesn't attempt to delete from a mounted system.
    if( oldExists( filename ) ) then
        oldDelete( filename );
    else
        for _, system in LuaVFS.Mounted do
            if( system:exists( filename ) ) then
                return system:deleteFile( filename );
            end
        end
    end
end

-- Write.
if( not oldWrite ) then
    oldWrite = _file.Write;
end
function _file.Write( filename, data, filesystem )
    if( not filesystem ) then
        oldWrite( filename, data );
    else
        filesystem:writeFile( filename, data );
    end
end

if( not oldFind ) then
    oldFind = _file.Find;
end
function _file.Find( filename )
    -- Table of found files.
    local ret = oldFind( filename );
    
    -- Run find on each system, and append files if they don't exist.
    for _, system in LuaVFS.Mounted do
        local sysret = system:find( filename );
        for _, file in sysret do
            local exists = false;
            for _, file2 in ret do
                if( string.lower( file2 ) == string.lower( file ) ) then
                    exists = true;
                end
            end
            
            if( not exists ) then
                table.insert( ret, file );
            end
        end
    end
    
    return ret;
end

if( not oldRename ) then
    oldRename = _file.Rename;
end
function _file.Rename( filename, newfilename )
    -- Check if this file exists, if it does read normally, if it doesn't attempt to read from a mounted system.
    if( oldExists( filename ) ) then
        oldRename( filename, newfilename );
    else
        for _, system in LuaVFS.Mounted do
            if( system:exists( filename ) ) then
                return system:renameFile( filename, newfilename );
            end
        end
    end
end

if( not oldCreate ) then
    oldCreate = _file.CreateDir;
end
function _file.CreateDir( dir, filesystem )
    if( not filesystem ) then
        oldCreate( dir );
    else
        local segments = string.blowup( filesystem:_fixPath( dir ), "/" );
        if( table.getn( segments ) <= 0 ) then
            table.insert( segments, filesystem:_fixPath( dir ) );
        end
        
        local curdir = segments[1];
        filesystem:addDirectory( curdir );
        if( table.getn( segments ) > 1 ) then
            for i = 2, table.getn( segments ) do
                curdir = curdir .. "/" .. segments[i];
                filesystem:addDirectory( curdir );
            end
        end
    end
end

-- Internal methods.
--------------------------------------------
function LuaVFS:_readFileSystem( filename )
    -- Load a file system.
    local buffer = oldRead( filename );
    
    -- local magic position.
    local mpos = string.find( buffer, "\t****\t" );
    local headers = string.blowup( string.sub( buffer, 1, mpos - 1 ), "\n" );
    if( table.getn( headers ) > 1 ) then
        table.remove( headers, table.getn( headers ) );
    end
    local data = string.sub( buffer, mpos + 6 );
    
    -- Parse files.
    for _, header in headers do
        -- Seperate information.
        local header_data = string.blowup( header, "|" );
        
        -- Get file size, and location in buffer.
        local size = tonumber( header_data[1] );
        local start = tonumber( header_data[2] );
        
        -- Get filename.
        local filename = header_data[3];
        
        -- Get directory.
        local is_dir = false;
        if( tonumber( header_data[4] ) == 1 ) then
            is_dir = true;
        end
        
        -- Get the actuall file data.
        local file_data = "";
        if( not is_dir ) then
            file_data = string.sub( data, start, start + ( size - 1 ) );
        end
        
        -- Construct the file.
        local file = { isdir = is_dir, name = filename, size = size, buffer = file_data };
        
        -- Insert into our file list.
        table.insert( self.files, file );
    end
end

function LuaVFS:_saveFileSystem( filename )
    -- Construct a buffer of data.
    local header_buffer = "";
    local file_buffer = "";
    
    -- Buffer position.
    local buffer_pos = 1;
    
    -- Get all information.
    for _, file in self.files do
        -- Write header.
        local dir = 0;
        if( file.isdir ) then
            dir = 1;
        end
        header_buffer = header_buffer .. string.len( file.buffer ) .. "|" .. buffer_pos .. "|" .. file.name .. "|" .. dir .. "\n";
        
        -- Write data.
        file_buffer = file_buffer .. file.buffer;
        
        -- Step the position.
        if( not file.isdir ) then
            buffer_pos = buffer_pos + string.len( file.buffer );
        end
    end
    
    -- Write system.
    _file.Write( filename, header_buffer .. "\t****\t" .. file_buffer );
end

function LuaVFS:_fixPath( path )
    if( string.sub( path, string.len( path ) - 1 ) == "\\" or string.sub( path, string.len( path ) - 1 ) == "/" ) then
        path = string.sub( path, 1, string.len( path ) - 1 );
    end
    
    return string.gsub( path, "\\", "//" );
end

function LuaVFS:_findFile( filename )
    -- Loop through all files, and search for the given one.
    for key, file in self.files do
        if( string.lower( self:_fixPath( filename ) ) == string.lower( self:_fixPath( file.name ) ) ) then
            return file, key;
        end
    end
    
    -- No file found, nil.
    return nil, nil;
end

-- Public methods.
--------------------------------------------

-- Init a new file system.
function LuaVFS:new( filename )
    -- Make object.
    local obj = {};
    setmetatable( obj, self );
    self.__index = self;
    
    -- Init system.
    obj.files = {};
    obj.from_internal = false;
    obj.filename = filename;
    
    -- If this vfs exists, open it and read the data.
    if( oldExists( filename ) ) then
        obj:_readFileSystem( filename );
    end
    
    -- Mount this system.
    table.insert( LuaVFS.Mounted, obj );
    
    return obj;
end

-- Get file count.
function LuaVFS:getFileCount( )
    local count = 0;
    for _, file in self.files do
        if( not file.isdir ) then
            count = count + 1;
        end
    end
    return count;
end

-- Unmount.
function LuaVFS:unmountFileSystem( )
    local idx = 1;
    while( idx <= table.getn( LuaVFS.Mounted ) ) do
        if( self:_fixPath( LuaVFS.Mounted[idx].filename ) == self:_fixPath( self.filename ) ) then
            table.remove( LuaVFS.Mounted, idx );
        else
            idx = idx + 1;
        end
    end
end

-- Get files.
function LuaVFS:getFiles( )
    -- Construct a list of filenames.
    local files = {};
    for _, file in self.files do
        if( not file.isdir ) then
            table.insert( files, file.name );
        end
    end
    return files;
end

-- Read file.
function LuaVFS:readFile( filename )
    -- Search for file.
    local file = self:_findFile( filename );
    if( file ) then
        -- Return its data.
        return file.buffer;
    end
    
    -- Not found return nothing.
    _Msg( "[LuaVFS] File not found.\n" );
    return "";
end

-- rename file.
function LuaVFS:renameFile( filename, newfilename )
    -- Search for file.
    local file = self:_findFile( filename );
    if( file ) then
        file.name = self:_fixPath( newfilename );
    end
    
    -- Not found return nothing.
    _Msg( "[LuaVFS] File not found.\n" );
end

-- Exists
function LuaVFS:exists( filename )
    -- Search for file.
    local file = self:_findFile( filename );
    if( file ) then
        return true;
    end
    
    return false;
end

-- Open script.
function LuaVFS:openScript( filename )
    -- Search for file.
    local file = self:_findFile( filename );
    if( file ) then
        _RunString( file.buffer );
    end
    
    -- Not found return nothing.
    _Msg( "[LuaVFS] File not found.\n" );
    return "";
end

-- Write file.
function LuaVFS:writeFile( filename, data )
    -- Search for file.
    local file = self:_findFile( filename );
    if( file ) then
        -- Write into its buffer.
        file.size = string.len( data );
        file.buffer = data;
    else
        -- Create a new file and write into its buffer.
        local file = { isdir = false, name = self:_fixPath( filename ), size = string.len( data ), buffer = data };
        
        -- Insert into our file list.
        table.insert( self.files, file );
    end
end

-- Add file.
function LuaVFS:addFile( filename )
    -- Add a file to the system.
    if( oldExists( self:_fixPath( filename ) ) ) then
        -- Does this file already exist within this system?
        local file = self:_findFile( filename );
        if( file ) then
            _Msg( "[LuaVFS] This file already exists in this system.\n" );
            return;
        else
            -- Construct the file.
            local data = _file.Read( filename );
            local file = { isdir = false, name = self:_fixPath( filename ), size = string.len( data ), buffer = data };
            
            -- Insert into our file list.
            table.insert( self.files, file );
        end
    end
end

-- Add directory.
function LuaVFS:addDirectory( path )
    -- Does this file already exist within this system?
    local file = self:_findFile( path );
    if( file ) then
        _Msg( "[LuaVFS] This directory already exists in this system.\n" );
        return;
    else
        -- Construct the dir.
        local file = { isdir = true, name = self:_fixPath( path ), size = 0, buffer = "" };
        
        -- Insert into our file list.
        table.insert( self.files, file );
    end
end

-- get directories.
function LuaVFS:getDirectories( )
    -- Construct a list of filenames.
    local files = {};
    for _, file in self.files do
        if( file.isdir ) then
            table.insert( files, file.name );
        end
    end
    return files;
end

-- Some nice general find functions.
function LuaVFS:find( pPath )
    -- Pad path.
    local path = self:_fixPath( pPath );
    if( not string.find( pPath, "/" ) ) then
        path = "/" .. pPath;
    end
    
    -- Filter.
    local filter = string.blowup( string.sub( path, string.lastfind( path, "/" ) + 1 ), "%." );

    -- Chop filter out of path.
    path = string.sub( path, 1, string.lastfind( path, "/" ) );
    
    -- Return table.
    local ret = {};
    
    -- Loop through the files in the system, searching for one that contains path/filter.filter.
    for _, file in self.files do
        -- Seperate the folder from the main path.
        local _path = "";
        local _file = file.name;
        if( string.find( file.name, "/" ) ) then
            _path = string.sub( file.name, 1, string.lastfind( file.name, "/" ) );
            _file = string.sub( file.name, string.lastfind( file.name, "/" ) + 1 );
        end
        
        -- Does _path match?
        if( string.lower( _path ) == string.lower( path ) ) then
            if( not file.isdir ) then
                -- Test filename againsted filter.
                if( table.getn( filter ) > 1 ) then
                    -- There is a filename filter and a file extension filter.
                    local fname = string.sub( _file, 1, string.lastfind( _file, "%." ) - 1 );
                    local fext = string.sub( _file, string.lastfind( _file, "%." ) + 1 );
                    
                    if( file.isdir or string.lower( _file ) == string.lower( string.gsub( filter[1], "*", fname ) .. "." .. string.gsub( filter[2], "*", fext ) ) ) then
                        table.insert( ret, _file ); 
                    end
                elseif( table.getn( filter ) > 0 ) then
                    -- Wild card, return everything.
                    if( string.lower( _file ) == string.lower( string.gsub( filter[1], "*", _file ) ) ) then
                        table.insert( ret, _file ); 
                    end
                else
                    -- No filter, just return everything.
                    table.insert( ret, _file );
                end
            else
                table.insert( ret, _file );
            end
        end
    end
    
    return ret;
end

-- Delete file.
function LuaVFS:deleteFile( filename )
    -- Delete a file from the system.
    local file, key = self:_findFile( filename );
    if( file ) then
        table.remove( self.files, key );
    else
        _Msg( "[LuaVFS] File not found.\n" );
    end
end

-- Save file system.
function LuaVFS:saveFileSystem( )
    -- If there is a filename. write.
    if( string.len( self.filename ) > 0 ) then
        self:_saveFileSystem( self.filename );
    else
        _Msg( "[LuaVFS] Attempted to write file system with no filename.\n" );
    end
end

-- Is directory?
function LuaVFS:isDirectory( dir )
    -- Search for file.
    local file = self:_findFile( dir );
    if( file and file.isdir ) then
        return true;
    end
    return false;
end

-- Utility Functions.
--------------------------------------------
function string.blowup( stringtoblowup, delimeter )
    local ret = {};
    local buffer = stringtoblowup;
    
    while( string.find( buffer, delimeter ) ~= nil ) do
        local s, e = string.find( buffer, delimeter );
        table.insert( ret, string.sub( buffer, 1, s - 1 ) );
        buffer = string.sub( buffer, e + 1 );
    end
    
    if( string.len( buffer ) > 0 ) then
        table.insert( ret, buffer );
    end
    
    return ret;
end

function string.lastfind( str, find, start )
    local pos = string.find( str, find, start or 1 );
    local lastvalid = nil;
    while( pos ~= nil ) do
        lastvalid = pos;
        pos = string.find( str, find, pos + 1 );
    end
    return lastvalid;
end
--------------------------------------------

-- Convert Directory.
-- Recursivly searches the target directory and adds all files to the
-- system.
function LuaVFS:convertDirectory( dir )
    local ndir = self:_fixPath( dir );
    
    -- Add files and other directories.
    local files = oldFind( ndir .. "/*.*" );
    for _, file in files do
        if( not _file.IsDir( ndir .. "/" .. file ) ) then
            self:addFile( ndir .. "/" .. file );
        else
            if( file ~= "." and file ~= ".." ) then
                self:addDirectory( ndir .. "/" .. file );
                self:convertDirectory( ndir .. "/" .. file );
            end
        end
    end
end

-- Extract.
function LuaVFS:extractFiles( dir )
    local ndir = self:_fixPath( dir );
    for _, file in self.files do
        if( file.isdir ) then
            _file.CreateDir( file.name );
        else
            -- Get directory.
            local directory = string.sub( file.name, 1, string.lastfind( file.name, "/" ) );
            
            -- Create the directory if it does not exist.
            _file.CreateDir( ndir .. "/" .. directory );
            
            -- Write file.
            _file.Write( file.name, file.buffer );
        end
    end
end
