-- trim - Removes leading or trailing whitespace
function trim ( s )
	return string.gsub( s, "^%s*(.-)%s*$", "%1" )
end

-- stripQuotes - Used to strip quotes from strings
function stripQuotes( s )
	return string.gsub( s, "^[\"]*(.-)[\"]*[\r]?[\n]?$", "%1" )
end


-- splitArgs - Used for splitting console arguments, will obey quotation marks
function splitArgs( args )
	local argv = {}
	local argc = 0
	local quoteFlag = false
	local strStore = ""
	
	for a in string.gfind( args, "[%a%p%d]+") do
		if a ~= nil then
			if quoteFlag == false then
				if string.sub( a, 1, 1 ) == "\"" and string.sub( a, -1, -1 ) ~= "\"" then -- If we're beginning a quote
					quoteFlag = true
					strStore = a -- Since this is the beginning, assign the store the beginning
				else
					a = stripQuotes( a )
					table.insert( argv, a )
					argc = argc + 1
				end
			else -- If quoteFlag is true
				if string.sub( a, -1, -1 ) == "\"" then -- If we're ending our quote
					quoteFlag = false
					strStore = strStore .. " " .. a
					strStore = stripQuotes( strStore )
					table.insert( argv, strStore )
					argc = argc + 1
				else
					strStore = strStore .. " " .. a -- Add to our store
				end
			end
		end
	end
	
	return argv, argc
end

function commandToFnName( command )
	local fn_name = "ccULX_" .. string.gsub( command, "%-", "M_" )
	fn_name = string.gsub( fn_name, "%+", "P_" )
	fn_name = string.gsub( fn_name, " ", "_" )
	
	return fn_name
end

-- Explodes strings, this comes from lua-users.org. Thank you!
function explode( d, p )
	t = {}
	local ll = 0
	while true do
		local l = string.find( p, d, ll + 1, true ) -- find the next d in the string
		if l ~= nil then -- if "not not" found then..
			table.insert( t, string.sub( p, ll, l - 1 ) ) -- Save it in our array.
			ll = l + 1 -- save just after where we found it for searching next time.
		else
			table.insert( t, string.sub( p, ll ) ) -- Save what's left in our array.
			break -- Break at end, as it should be, according to the lua manual.
		end
	end
	return t
end

-- print_functioncall - Pass this the name of the function in a string and 
-- all the arguments of a function to get a helpful debug output of the function call
-- EXAMPLE OUTPUT:
-- [ULX] Debug function call: sendTextAndAnimate( (number)1, (number)10, 
-- (number)0, (number)0.3, (number)0.05, (number)0.51, (number)255, (number)255, 
-- (number)255, (string)7. Exit, (string)ChatFont, (number)60, (number)0.5, (number)0.5, 
-- (number)0.5, (number)0.3,  )
function print_functioncall( function_name, ... )
	_Msg( "[ULX] Debug function call: " .. function_name .. "( " )
	for i=1, table.getn( arg ) do
		_Msg( "(" .. type( arg[ i ] ) .. ")" .. tostring( arg[ i ] ) .. ", " )
	end
	_Msg( " )\n" )
end

function debug( text )
	--if DEBUG_MODE then
		_Msg( "[ULX] Debug: " .. text .. "\n" )
	--end
end

function ulx_debug( text )
	return debug( text )
end

function error_msg( text )
	_Msg( "[ULX] ERROR: " .. text .. "\n" )
end

function ulx_log( text )
 -- ADD timestamp when available
	_file.Write( "lua/ulx/logs/log.txt", _file.Read( "lua/ulx/logs/log.txt" ) .. text .. "\r\n" )
end

function ulx_playerString( userid )
	if _PlayerInfo( userid, "connected" ) == false then
		return "<NOT CONNECTED>"
	end
	
	return _PlayerInfo( userid, "name" ) .. "(steamid=" .. _PlayerInfo( userid, "networkid" ) .. ")"
end

-- sendText - sends the text in the specified format ( used for menus )
function sendText( userid, key, posx, posy, r, g, b, text, font, timeout, fadein, fadeout )
	_GModText_Start( font )
		_GModText_SetPos( posx, posy )
		_GModText_SetColor( r, g, b, 255 )
		_GModText_SetTime( timeout, fadein, fadeout )
		_GModText_SetText( text )
		_GModText_SetDelay(0)
		_GModText_AllowOffscreen( false )
	_GModText_Send( userid, key )
end

-- sendTextAnimate - animates the text ( you have to define the text with sendText or sendTextAndAnimate first! )
function sendTextAnimate( userid, key, posx, posy, r, g, b, font, length, ease )
	_GModText_Start( font )
		_GModText_SetPos( posx, posy )
		_GModText_SetColor( r, g, b, 255 )
		_GModText_AllowOffscreen( false )
	_GModText_SendAnimate( userid, key, length, ease )
end

-- sendTextAndAnimate - sends and animates the text in the specified format ( used for menus )
function sendTextAndAnimate( userid, key, begPosx, begPosy, posx, posy, r, g, b, text, font, timeout, fadein, fadeout, length, ease )
	_GModText_Start( font )
		_GModText_SetPos( begPosx, begPosy )
		_GModText_SetColor( r, g, b, 255 )
		_GModText_SetTime( timeout, fadein, fadeout )
		_GModText_SetText( text )
		_GModText_SetDelay( 0 )
		_GModText_AllowOffscreen( false )
	_GModText_Send( userid, key )
	
		
	_GModText_Start( font );
		_GModText_SetPos( posx, posy )
		_GModText_SetColor( r, g, b, 255 )
		_GModText_AllowOffscreen( false )
	_GModText_SendAnimate( userid, key, length, ease );
end

-- sendRect - creates a rectangle in the specified format ( used for menus )
function sendRect( userid, key, rectBegX, rectBegY1, rectBegSizeX, rectBegSizeY, r, g, b, a, material, timeout, fadein, fadeout )
	_GModRect_Start( material )
		_GModRect_SetPos( rectBegX, rectBegY1, rectBegSizeX, rectBegSizeY )
		_GModRect_SetColor( r, g, b, a )
		_GModRect_SetTime( timeout, fadein, fadeout )
		_GModRect_SetDelay( 0 )
	_GModRect_Send( userid, key )
end

-- sendRectAnimate - animates the rectangle ( you have to define the text with sendRect or sendRectAndAnimate first! )
function sendRectAnimate( userid, key, rectEndX, rectEndY, rectEndSizeX, rectEndSizeY, r, g, b, a, material, length, ease )
	_GModRect_Start( material );
		_GModRect_SetPos( rectEndX, rectEndY, rectEndSizeX, rectEndSizeY )
		_GModRect_SetColor( r, g, b, a )
	_GModRect_SendAnimate( userid, key, length, ease )
end

-- sendRectAndAnimate - sends and animates the rectangle in the specified format ( used for menus )
function sendRectAndAnimate( userid, key, rectBegX, rectBegY1, rectBegSizeX, rectBegSizeY, rectEndX, rectEndY, rectEndSizeX, rectEndSizeY, r, g, b, a, material, timeout, fadein, fadeout, length, ease )
	_GModRect_Start( material )
		_GModRect_SetPos( rectBegX, rectBegY1, rectBegSizeX, rectBegSizeY )
		_GModRect_SetColor( r, g, b, a )
		_GModRect_SetTime( timeout, fadein, fadeout )
		_GModRect_SetDelay( 0 )
	_GModRect_Send( userid, key )
	
	_GModRect_Start( material );
		_GModRect_SetPos( rectEndX, rectEndY, rectEndSizeX, rectEndSizeY )
		_GModRect_SetColor( r, g, b, a )
	_GModRect_SendAnimate( userid, key, length, ease )
end

function ulx_play3DSound( sound, vector, volume, pitch )
	volume = volume or 1
	pitch = pitch or 1
	
	local newent = _EntCreate( "info_null" )
	_EntSetPos( newent, vector )
	_EntSpawn( newent )
	
	_EntEmitSoundEx( newent, sound, volume, pitch )	
end

-- These are the sounds that will be used for slaps
gSlapSounds = {
"physics/body/body_medium_impact_hard1.wav",
"physics/body/body_medium_impact_hard2.wav",
"physics/body/body_medium_impact_hard3.wav",
"physics/body/body_medium_impact_hard5.wav",
"physics/body/body_medium_impact_hard6.wav",
"physics/body/body_medium_impact_soft5.wav",
"physics/body/body_medium_impact_soft6.wav",
"physics/body/body_medium_impact_soft7.wav"
}

function ulx_slap( target, damage, power )	
	-- Play a slap sound
	local sound_num = math.random( table.getn( gSlapSounds ) ) -- Choose at random
	ulx_play3DSound( gSlapSounds[ sound_num ], _EntGetPos( target ), 1.5 )
	
	local vec = { x=math.random( 20 )-10, y=math.random( 20 )-10, z=math.random( 20 )-8 }
	vec = vecNormalize( vec )  -- This should make the slaps more equal in force, no matter what the direction
	vec = vecMul( vec, vector3( power, power, power ) ) -- Now make the slap stronger
	
	_EntSetVelocity( target, vec )
	
	local newHp = _PlayerInfo( target, "health" ) - damage
		
	if newHp <= 0 then
		_PlayerKill( target )
		return
	end
		
	_PlayerSetHealth( target, newHp )
end

-- This will log the message to the console as well as show it in the talk area
function ulx_tsay( userid, msg )
	_PrintMessage( userid, HUD_PRINTTALK, msg .. "\n" )
	_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )	
end

-- This will log the message to the console as well as show it in the talk area ( parses special words )
function ulx_tsay_parse( userid, msg )
	msg = string.gsub( msg, "%%hostname%%", _GetConVar_String( "hostname" ) )
	msg = string.gsub( msg, "%%curmap%%", _GetCurrentMap() )
	msg = string.gsub( msg, "%%nextmap%%", _GetNextMap() )	
	msg = string.gsub( msg, "%%ulx_version%%", ULX_VERSION )
	msg = string.gsub( msg, "%%playername%%", _PlayerInfo( userid, "name" ) )
	msg = string.gsub( msg, "%%playersteam%%", _PlayerInfo( userid, "networkid" ) )
	msg = string.gsub( msg, "%%playerping%%", _PlayerInfo( userid, "ping" ) )
	msg = string.gsub( msg, "%%playerkills%%", _PlayerInfo( userid, "kills" ) )	
	msg = string.gsub( msg, "%%playerdeaths%%", _PlayerInfo( userid, "deaths" ) )		
		
	ulx_tsay( userid, msg )
end

-- This will log the message to the console as well as show it in the talk area
function ulx_tsay_all( msg )
	_PrintMessageAll( HUD_PRINTTALK, msg .. "\n" )
	_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )	
end

-- This will log the message to the console as well as show it in the talk area ( parses special words )
function ulx_tsay_all_parse( msg )
	for i=1, 32 do
		if _PlayerInfo( i, "connected" ) == true then
			ulx_tsay_parse( i, msg )
		end
	end
end

-- This will log the message to the console as well as show it in the center
function ulx_csay( userid, msg, r, g, b, length )
	r = r or 255
	g = g or 255
	b = b or 255
	length = length or 5
	sendText( userid, newTextKey(), -1, 0.35, r, g, b, msg .. "\n", "TargetID", length, 0.5, 0.5 )
	_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )	
end

-- This will log the message to the console as well as show it in the center ( parses special words )
function ulx_csay_parse( userid, msg )
	msg = string.gsub( msg, "%%hostname%%", _GetConVar_String( "hostname" ) )
	msg = string.gsub( msg, "%%curmap%%", _GetCurrentMap() )
	msg = string.gsub( msg, "%%nextmap%%", _GetNextMap() )	
	msg = string.gsub( msg, "%%ulx_version%%", ULX_VERSION )	
	msg = string.gsub( msg, "%%playername%%", _PlayerInfo( userid, "name" ) )
	msg = string.gsub( msg, "%%playersteam%%", _PlayerInfo( userid, "networkid" ) )
	msg = string.gsub( msg, "%%playerping%%", _PlayerInfo( userid, "ping" ) )
	msg = string.gsub( msg, "%%playerkills%%", _PlayerInfo( userid, "kills" ) )	
	msg = string.gsub( msg, "%%playerdeaths%%", _PlayerInfo( userid, "deaths" ) )		
		
	ulx_csay( userid, msg )
end

-- This will log the message to the console as well as show it in the center
function ulx_csay_all( msg, r, g, b )
	r = r or 255
	g = g or 255
	b = b or 255
	length = length or 5
	sendText( 0, newTextKey(), -1, 0.35, r, g, b, msg .. "\n", "TargetID", length, 0.5, 0.5 )
	_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )	
end

-- This will log the message to the console as well as show it in the center ( parses special words )
function ulx_csay_all_parse( msg )
	for i=1, 32 do
		if _PlayerInfo( i, "connected" ) == true then
			ulx_csay_parse( i, msg )
		end
	end
end

-- This will log the message to the console
function ulx_console( userid, msg )
	_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )	
end

-- This will log the message to the console ( parses special words )
function ulx_console_parse( userid, msg )
	msg = string.gsub( msg, "%%hostname%%", _GetConVar_String( "hostname" ) )
	msg = string.gsub( msg, "%%curmap%%", _GetCurrentMap() )
	msg = string.gsub( msg, "%%nextmap%%", _GetNextMap() )	
	msg = string.gsub( msg, "%%ulx_version%%", ULX_VERSION )	
	msg = string.gsub( msg, "%%playername%%", _PlayerInfo( userid, "name" ) )
	msg = string.gsub( msg, "%%playersteam%%", _PlayerInfo( userid, "networkid" ) )
	msg = string.gsub( msg, "%%playerping%%", _PlayerInfo( userid, "ping" ) )
	msg = string.gsub( msg, "%%playerkills%%", _PlayerInfo( userid, "kills" ) )	
	msg = string.gsub( msg, "%%playerdeaths%%", _PlayerInfo( userid, "deaths" ) )		
		
	ulx_console( userid, msg )
end

-- This will log the message to the console
function ulx_console_all( msg )
	_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )
end

-- This will log the message to the console ( parses special words )
function ulx_console_all_parse( msg )
	for i=1, 32 do
		if _PlayerInfo( i, "connected" ) == true then
			ulx_console_parse( i, msg )
		end
	end
end

-- Thanks goes to andyvincent for inspiring getFilesFromDir
function getFilesFromDir( dir )
	if _file.Exists( dir ) == false or _file.IsDir( dir ) == false then
		error_msg( "getFilesFromDir() was passed an invalid directory!" )
	end
	
	local files = {}
	local dirs = {}
	
	local result = _file.Find( dir .. "/*" )
	
	for i=1, table.getn( result ) do
		if result[ i ] == "." or result[ i ] == ".." or result[ i ] == "readme.txt" then -- We just want to ignore these ( readme.txt crashes linux servers, reason unknown )
		else
			if _file.IsDir( dir .. "/" .. result[ i ] ) == true then
				table.insert( dirs, result[ i ] )
			elseif _file.Exists( dir .. "/" .. result[ i ] ) == true then -- Double check it's a file
				table.insert( files, result[ i ] )
			end
		end
	end
	
	return files, dirs
end

function getFilesFromDirRecursive( dir )
	if _file.Exists( dir ) == false or _file.IsDir( dir ) == false then
		error_msg( "getFilesFromDirRecursive() was passed an invalid directory!" )
		return
	end
	
	local files = {}
	
	local result = _file.Find( dir .. "/*" )
	
	for i=1, table.getn( result ) do
		if result[ i ] == "." or result[ i ] == ".." or result[ i ] == "readme.txt" then -- We just want to ignore these ( readme.txt crashes linux servers, reason unknown )
		else
			if _file.IsDir( dir .. "/" .. result[ i ] ) == true then
				local temp = getFilesFromDirRecursive( dir .. "/" .. result[ i ] )
				for i=1, table.getn( temp ) do
					table.insert( files, temp[ i ] )
				end
			elseif _file.Exists( dir .. "/" .. result[ i ] ) == true then -- Double check it's a file
				table.insert( files, dir .. "/" .. result[ i ] )
			end
		end
	end
	
	return files
end

-- This will return the file, with each line of the file an element in the table.
function readFile( file, comment )
	if _file.Exists( file ) == false then
		error_msg( "readFile was passed an invalid file!" )
		return nil
	end
	
	local text = _file.Read( file )
	local t = {}
	
	while 1 do -- We have a break in the loop
		local tmp
		if string.find( text, "\n" ) ~= nil then
			tmp = trim( string.gsub( text, "^(.-)[\r]?\n(.*)$", "%1" ) )
			text = string.gsub( text, "^(.-)[\r]?\n(.*)$", "%2" )
		else
			--debug( "no new line" )
			if comment ~= nil and string.find( text, comment ) then
				local strBeg = string.find( text, comment )
				if strBeg == 1 then
					text = ""
				else
					text = trim( string.sub( text, 1, strBeg - 1 ) )
				end
			end
			
			if text ~= "" then
				table.insert( t, text )
			end
			break
		end
			
		if comment ~= nil and string.find( tmp, comment ) then
				local strBeg = string.find( tmp, comment )
				if strBeg == 1 then
					tmp = ""
				else
					tmp = trim( string.sub( tmp, 1, strBeg - 1 ) )
				end
		end			

		if tmp ~= "" then
			table.insert( t, tmp )
		end
	end
	return t
end

-- Writes a table as a table, each element on one line
function writeFile( t, file )
	local text = ""
	for i=1, table.getn( t ) do
		text = text .. t[ i ] .. "\r\n"
	end
	_file.Write( file, text )
end

function execFile( file )
	if _file.Exists( file ) == false then
		error_msg( "execFile was passed an invalid file!" )
		return nil
	end	
	
	lines = readFile( file, "//" )

	for i=1, table.getn( lines ) do
		local command = string.sub( lines[ i ], 1, string.find( lines[ i ], " " ) - 1 )
		local args = trim( string.sub( lines[ i ], string.find( lines[ i ], " " ) + 1 ) )
	
		if gCvarHelp[ command ] ~= nil then -- This is a cvar!
			if gCvarHelp[ command ] == table then -- This is a client cvar
				_G[ command ].default = args
			else
				_G[ command ] = args
			end
		elseif gConsoleCommands[ command ] ~= nil then -- This is a lua command!
			if gCmdHelp[ command ] ~= nil then -- This is an ULX command!
				local argv, argc = splitArgs( args )
				gConsoleCommands[ command ]( 0, args, argv, argc ) -- Note that 0 userid may cause problems with some functions
			else
				gConsoleCommands[ command ]( 0, args ) -- Note that 0 userid may cause problems with some functions
			end
		else -- We're assuming it's just an average command to run
			_ServerCommand( lines[ i ] .. "\n" )
		end
	end
end

-- addUser - Adds a user to our permanent access list
function addUser( steamID, access ) 
	gUsers[ steamID ] = access
	return
end


-- Strips a file's extension
function stripExt( file )
	return string.gsub( file, "^(.*)%..-$", "%1" )
end

function loadTable( file )
	if _file.Exists( "lua/" .. file ) == false then
		error_msg( "loadTable was passed an invalid file!" )
		return
	end	
	
	local tmp = _G[ "t" ] -- Preserve the global
	local file = _file.Read( "lua/" .. file )
	local f = loadstring( file )
	f()
	local t = _G[ "t" ]
	_G[ "t" ] = tmp
	return t
end

-- Save a table
function saveTable( t, file, comment )
	comment = comment or ""
	file = "lua/" .. file
	if _file.IsDir( file ) == true then
		error_msg( "saveTable was passed a directory when it needed a file!" )
		return
	end
	
	if type( t ) ~= "table" then
		error_msg( "saveTable was passed an invalid table!" )
		return
	end
	
	_file.Write( file, "-- This code was generated by ULX's table saver.\r\n-- " ..
		comment .. "\r\nt = {}\r\n\r\n" )
	saveTableRecursive( t, file, "t" )
end

-- DO NOT CALL THIS DIRECTLY, use saveTable() -- Thanks to http://www.gammon.com.au/forum/bbshowpost.php?bbsubject_id=4903 for inspiration
function saveTableRecursive( t, file, prefix, done )
	done = done or {}
	
	for k, v in pairs( t ) do
		local key = k
		local value = v
		
		if type( k ) == "string" then
			key = string.gsub( k, "\"", "\\\"" )
			key = "\"" .. key .. "\""
		end
		
		if type( v ) == "string" then
			value = string.gsub( v, "\"", "\\\"" )
			value = "\"" .. value .. "\""
		end
		
		if type( value ) == "table" and done[ value ] == nil then
			done [value] = true
			_file.Write( file, _file.Read( file ) .. prefix .. "[ " .. tostring( key ) .. " ]\t=\t{}\r\n" )			
			saveTableRecursive( value, file, "\t" .. prefix .. "[ " .. tostring( key ) .. " ]", done )
	    else
			_file.Write( file, _file.Read( file ) .. prefix .. "[ " .. tostring( key ) .. " ]\t=\t" .. tostring( value ) .. "\r\n" )
	    end
	end
end

-- Checks if two vectors are equal
function vecEqual( v1, v2 )	
	if  v1.x == v2.x and
		v1.y == v2.y and
		v1.z == v2.z then
		return true
	end
	
	return false
end

-- Generates res from our download info
function generateRes()
	local map = _GetCurrentMap()
	if _file.Exists( "maps/" .. map .. ".res" ) == false then
		_file.Write( "maps/" .. map .. ".res", "\"Resources\"\r\n{\r\n}" )
	end
	
	local res = readFile( "maps/" .. map .. ".res" )
	
	local scopeBeg = nil
	local scopeEnd = nil
	for i=1, table.getn( res ) do
		if string.find( string.lower( res[ i ] ), "\"resources\"" ) ~= nil then
			if string.find( res[ i ], "{" ) ~= nil then
				scopeBeg = i
				break
			elseif string.find( res[ i + 1 ], "{" ) then
				scopeBeg = i + 1
				break
			else
				error_msg( "ULX encountered an invalid .res file for map " .. map )
				break
			end
		end
	end
	
	for i=scopeBeg, table.getn( res ) do
		if res[ i ] == nil then
			break  -- We need this because we're removing from the table later
		end
		
		--debug( res[ i ] )
		if string.find( res[ i ], gForceDLKeyBeg ) ~= nil then -- Delete it, we're going to replace it
			for c=i, table.getn( res ) do
				if string.find( res[ c ], gForceDLKeyEnd ) ~= nil then
					for b=i, c do
						table.remove( res, i )
					end
					i = i - 1
					break
				end
			end
		end
		
		if string.find( res[ i ], "}" ) ~= nil then
			if string.sub( trim( res[ i ] ), 1, 1 ) == "}" then -- If it's on it's own line
				scopeEnd = i
				break
			end
			-- Otherwise, let's put it on it's own new line
			res[ i ] = string.gsub( res[ i ], "}", "" )
			table.insert( res, i + 1, "}" )
			scopeEnd = i + 1
			break
		end
	end
	
	if scopeBeg ~= nil and scopeEnd ~= nil then
		local forcedl = readFile( "lua/ulx/configs/forcedownload.ini", "//" )
		table.insert( res, scopeEnd, "\t" .. gForceDLKeyBeg )
		for i=1, table.getn( forcedl ) do
			table.insert( res, scopeEnd + i, "\t" .. forcedl[ i ] )
		end
		table.insert( res, scopeEnd + table.getn( forcedl ) + 1, "\t" .. gForceDLKeyEnd )		
	end
	
	--debug( scopeBeg )
	--debug( scopeEnd )
	--tprint( res )
	writeFile( res, "maps/" .. map .. ".res" )
end




-- This comes from lua.org, thanks for teaching millions lua, and providing this awesome code!
List = {}
function List.new ()
	return {first = 0, last = -1}
end


function List.pushleft (list, value)
	local first = list.first - 1
	list.first = first
	list[first] = value
end
    
function List.pushright (list, value)
	local last = list.last + 1
	list.last = last
	list[last] = value
end
    
function List.popleft (list)
	local first = list.first
	if first > list.last then error("list is empty") end
	local value = list[first]
	list[first] = nil        -- to allow garbage collection
	list.first = first + 1
	return value
end
    
function List.popright (list)
	local last = list.last
	if list.first > last then error("list is empty") end
	local value = list[last]
	list[last] = nil         -- to allow garbage collection
	list.last = last - 1
	return value
end

function List.isEmpty (list)
	if list.first > list.last then
		return true
	else
		return false
	end
end
