function gmx.isValid(value,reqType)
	if type(value)~="nil" then
		if reqType~=nil then
			if type(value)==string.lower(reqType) then
				return true;
			else
				return false;
			end
		else
			return true;
		end
	else
		return false;
	end
end

function gmx.allValid(values,reqTypes)
	if gmx.isValid(values,"table") then
		local allValid=true
		for value in values do
			if gmx.isValid(reqTypes,"table") then
				if gmx.isValid(values[value],reqTypes[value])==false then
					allValid=false;
				end
			else
				if gmx.isValid(values[value])==false then
					allValid=false;
				end
			end
		end
		return allValid
	else
		return false;
	end
end

function gmx.error(message)
	gmx.message(0,message,"GMX Error",2);
	return nil;
end

function gmx.strToBool(str)
	if gmx.isValid(str,"string")==false then return gmx.error("gmx.strToBool - Str is not valid") end
	if string.lower(str)=="true" then return true
	elseif string.lower(str)=="yes" then return true
	elseif string.lower(str)=="1" then return true
	elseif string.lower(str)=="on" then return true
	elseif string.lower(str)=="false" then return false
	elseif string.lower(str)=="no" then return false
	elseif string.lower(str)=="0" then return false
	elseif string.lower(str)=="off" then return false
	else return gmx.error("gmx.strToBool - String was not recognized as having any boolean words in it.")
	end
end

function gmx.boolToStr(bool,type)
	if gmx.allValid({bool,type},{"boolean","number"})==false then return gmx.error("gmx.boolToStr - Arguments invalid.") end
	if bool==true or bool==1 then
		if type==1 then return "1"
		elseif type==2 then return "true"
		elseif type==3 then return "yes"
		elseif type==4 then return "on"
		else return gmx.error("gmx.boolToStr - Type was not recognized.");
		end
	elseif bool==false or bool==0 then
		if type==1 then return "0"
		elseif type==2 then return "false"
		elseif type==3 then return "no"
		elseif type==4 then return "off"
		else return gmx.error("gmx.boolToStr - Type was not recognized.");
		end
	else return gmx.error("gmx.boolToStr - Boolean was neither true, 1, false, or 0");
	end
end

function gmx.explode(str,seperator,limit)
	if gmx.allValid({str,seperator},{"string","string"})==false then return gmx.error("gmx.explode - Arguments invalid.") end
	local t={};
	local lim=-1;
	local reps=0;
	local ll=0;
	if gmx.isValid(limit,"number") then lim=limit-1 end
	while true do
		local l=string.find(str,seperator,ll+1,true);
		if l~=nil then
			table.insert(t,string.sub(str,ll,l-1));
			ll=l+1;
			reps=reps+1;
		else
			table.insert(t,string.sub(str,ll));
			reps=reps+1;
			break;
		end
		
		if reps>=lim and lim~=-1 then
			table.insert(t,string.sub(str,ll));
			break;
		end
	end
	return t;
end 

function gmx.implode(pieces,glue)
	if gmx.allValid({pieces,glue},{"table","string"})==false then return gmx.error("gmx.implode - Arguments invalid.") end
	local fullString="";
	local first=true;
	
	for piece in pieces do
		if first then
			fullString = pieces[piece];
			first=false;
		else
			fullString = fullString .. glue .. pieces[piece];
		end
	end
	return fullString;
end

function gmx.compareNCC(strA,strB)
	if gmx.allValid({strA,strB},{"string","string"})==false then return gmx.error("gmx.compareNCC - Arguments invalid.") end
	return (string.lower(strA)==string.lower(strB));
end

function gmx.dataToKVStr(data,name,level)
	if gmx.allValid({data,name,level})==false then return gmx.error("gmx.dataToKVStr - Arguments invalid.") end
	if level>table.getn(gmx.seperators) then return gmx.error("gmx.KVStrToData - Table level is too deep to record!") end
	local abbvr="x";
	local val=data;
	local safeName=0;
	if gmx.isValid(name,"number") then safeName=name end
	if gmx.isValid(name,"string") then safeName=gmx.stripIllegalChars(name) end
	if type(data)=="string" then
		abbvr="s";
		val=gmx.stripIllegalChars(data);
	elseif type(data)=="number" then
		abbvr="n";
		val=""..data;
	elseif type(data)=="table" then
		abbvr="t";
		val=gmx.tableToStr(data,level+1);
	elseif type(data)=="boolean" then
		abbvr="b";
		val=gmx.boolToStr(data,1);
	elseif type(data)=="function" then
		abbvr="f";
		val="";
	elseif type(data)=="userdata" then
		if gmx.isVector(data) then
			abbvr="v";
			val=data.x..gmx.vecSep..data.y..gmx.vecSep..data.z;
		end
	else
		abbvr="x";
		val="";
	end
	return abbvr..gmx.kvSep..safeName..gmx.kvSep..val;
end

function gmx.KVStrToData(str,useTable,level)
	if gmx.allValid({str,useTable,level},{"string","table","number"})==false then return gmx.error("gmx.KVStrToData - Arguments invalid.") end
	if level>table.getn(gmx.seperators) then return gmx.error("gmx.KVStrToData - Table level is too deep to get!") end
	local pieces=gmx.explode(str,gmx.kvSep,3);
	local newTable=useTable;
	local indexNum=tonumber(pieces[2]);
	if indexNum~=nil then pieces[2]=indexNum end
	if pieces[1]=="s" then
		newTable[pieces[2]]=pieces[3];
	elseif pieces[1]=="n" then
		newTable[pieces[2]]=tonumber(pieces[3]);
	elseif pieces[1]=="t" then
		newTable[pieces[2]]=gmx.strToTable(pieces[3],level+1);
	elseif pieces[1]=="b" then
		newTable[pieces[2]]=gmx.strToBool(pieces[3]);
	elseif pieces[1]=="v" then
		local vecList=gmx.explode(pieces[3],gmx.vecSep,3);
		local vX=tonumber(vecList[1]);
		local vY=tonumber(vecList[2]);
		local vZ=tonumber(vecList[3]);
		newTable[pieces[2]]=vector3(vX,vY,vZ);
	elseif pieces[1]=="f" then
		newTable[pieces[2]]="[function]";
	elseif pieces[1]=="x" then
		newTable[pieces[2]]="[unknown]"
	end
	return newTable;
end

function gmx.tableToStr(useTable,level)
	if gmx.allValid({useTable,level},{"table","number"})==false then return gmx.error("gmx.tableToStr - Arguments invalid.") end
	if level>table.getn(gmx.seperators) then return gmx.error("gmx.KVStrToData - Table level is too deep to record!") end
	local pieces={};
	for key,value in useTable do
		pieces[key]=gmx.dataToKVStr(value,key,level);
	end
	local fullString=gmx.implode(pieces,gmx.seperators[level]);
	return fullString;
end

function gmx.strToTable(str,level)
	if gmx.allValid({str,level},{"string","number"})==false then return gmx.error("gmx.strToTable - Arguments invalid.") end
	if level>table.getn(gmx.seperators) then return gmx.error("gmx.KVStrToData - Table level is too deep to get!") end
	local pieces=gmx.explode(str,gmx.seperators[level]);
	local returnedTable={};
	for key,value in pieces do
		returnedTable=gmx.KVStrToData(value,returnedTable,level);
	end
	return returnedTable;
end

function gmx.saveTable(tableToUse,filename)
	if gmx.allValid({tableToUse,filename},{"table","string"})==false then return gmx.error("gmx.saveTable - Arguments invalid.") end
	_file.Write(filename,gmx.tableToStr(tableToUse,1));
end

function gmx.loadTable(filename)
	if gmx.isValid(filename,"string")==false then return gmx.error("gmx.loadTable - Arguments invalid.") end
	if _file.Exists(filename)==false then return nil end
	local fContents=_file.Read(filename);
	local finalTable=gmx.strToTable(fContents,1);
	return finalTable;
end

function gmx.tableDepth(depthTable)
	if gmx.isValid(depthTable,"table")==false then return gmx.error("gmx.loadTable - Arguments invalid.") end
	local highestDepth=0;
	for key,value in depthTable do
		if gmx.isValid(value,"table") then
			local childDepth=gmx.tableDepth(value);
			if childDepth>highestDepth then highestDepth=childDepth end
		end
	end
	return 1+highestDepth;
end

function gmx.tableMem(specTable,addSpaces)
	if gmx.isValid(specTable,"table")==false then return gmx.error("gmx.tableMem - Arguments invalid.") end
	local addSpacesLim=addSpaces or 0;
	local spaces=""
	local itemCount=0;
	for i=1,addSpacesLim do
		spaces=spaces.." ";
	end
	for k,v in specTable do
		itemCount=itemCount+1;
	end
	gmx.message(1,spaces.."("..itemCount.." items)")
	for key,value in specTable do
		if type(value)=="string" then
			gmx.message(1,spaces..key..": \""..value.."\"");
		elseif type(value)=="number" then
			gmx.message(1,spaces..key..": "..value);
		elseif type(value)=="boolean" then
			gmx.message(1,spaces..key..": "..gmx.boolToStr(value,2));
		elseif type(value)=="function" then
			gmx.message(1,spaces..key..": [function]",nil,2);
		elseif type(value)=="userdata" then
			if gmx.isVector(value) then
				gmx.message(1,spaces..key..": [vector] ("..gmx.implode(gmx.vectorToTable(value),",")..")",nil,2);
			else
				gmx.message(1,spaces..key..": [userdata]",nil,2);
			end
		elseif type(value)=="table" then
			gmx.message(1,spaces..key..": [table]",nil,2);
			gmx.tableMem(value,addSpacesLim+4)
		elseif type(value)=="nil" then
			gmx.message(1,spaces..key..": [nil]",nil,2);
		end
	end
end

function gmx.stripIllegalChars(str)
	if gmx.isValid(str,"string")==false then return gmx.error("gmx.stripIllegalChars - Arguments invalid.") end
	local returnedStr=str;
	returnedStr=string.gsub(returnedStr,gmx.kvSep,"");
	for char in gmx.seperators do
		returnedStr=string.gsub(returnedStr,gmx.seperators[char],"");
	end
	return returnedStr;
end

--Take a table and return a new table with items that meet the condition
function gmx.filter(useTable,conditionFunc)
	local filteredTable={};
	for k,v in useTable do
		if conditionFunc(k,v)==true then
			filteredTable[k]=v;
		end
	end
	return filteredTable;
end

--Returns true if table contains item and false otherwise.
function gmx.tableContains(useTable,item)
	for k,v in useTable do
		if v==item then
			return true;
		end
	end
	return false;
end

function gmx.addNamedThinkFunc(name,func)
	gLuaThinkFunctions[name]=func;
end

function gmx.addNamedTimer(name,delay,reps,func,...)
	Timer[name]={};
	Timer[name].delay=delay;
	Timer[name].time=_CurTime();
	Timer[name].func=func;
	Timer[name].arg=arg;
	Timer[name].reps=reps;
	return name;
end

function gmx.capFirstLetter(str)
	local firstLetter=string.upper(string.sub(str,1,1));
	local restOfString=string.sub(str,2,-1);
	
	return firstLetter..restOfString;
end

function gmx.capFirstAll(str)
	local words=gmx.explode(str," ");
	for k,v in words do
		words[k]=gmx.capFirstLetter(v);
	end
	return gmx.implode(words," ");
end

function gmx.randomElement(useTable)
	if gmx.isValid(useTable,"table")==false then return gmx.error("gmx.randomElement - Arguments invalid.") end
	return useTable[math.random(0,table.getn(useTable))];
end